#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2025, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'


from functools import partial

from qt.core import QInputDialog, QTabWidget, pyqtSignal

from calibre.gui2 import gprefs
from calibre.gui2.dialogs.template_dialog import TemplateDialog
from calibre.gui2.preferences import LazyConfigWidgetBase
from calibre.gui2.preferences.look_feel_tabs.bookshelf_view_ui import Ui_bookshelf_tab as Ui_Form


class BookshelfTab(QTabWidget, LazyConfigWidgetBase, Ui_Form):

    changed_signal = pyqtSignal()
    restart_now = pyqtSignal()

    def __init__(self, parent=None):
        super().__init__(parent)

    def genesis(self, gui):
        self.gui = gui
        db = self.gui.current_db
        r = self.register

        r('bookshelf_shadow', gprefs)
        r('bookshelf_thumbnail', gprefs)
        r('bookshelf_variable_height', gprefs)
        r('bookshelf_fade_time', gprefs)

        r('bookshelf_title_template', db.prefs)
        r('bookshelf_spine_size_template', db.prefs)

        self.config_cache.link(
            self.gui.bookshelf_view.cover_cache,
            'bookshelf_disk_cache_size', 'bookshelf_cache_size_multiple',
        )
        self.opt_bookshelf_spine_size_template.setToolTip(_('''
<p>The template used to calculate a width for the displayed spine.
The template must evaluate to a decimal number between 0.0 and 1.0, which will be used to set the width of the books spine.
An empty template means a fixed spine size for all books.
The special template {0} uses the book size to estimate a spine size.
The special template {1} uses a random size.
You can also use a number between 0.0 and 1.0 to pick a fixed size.
<p>
Note that this setting is per-library, which means that you have to set it again for every
different calibre library you use.</p>''').format('{size}', '{random}'))

        self.template_title_button.clicked.connect(partial(self.edit_template_button, self.opt_bookshelf_title_template))
        self.template_pages_button.clicked.connect(partial(self.edit_template_button, self.opt_bookshelf_spine_size_template))
        self.use_pages_button.clicked.connect(self.use_pages)

    def edit_template_button(self, line_edit):
        rows = self.gui.library_view.selectionModel().selectedRows()
        mi = None
        db = self.gui.current_db.new_api
        if rows:
            ids = list(map(self.gui.library_view.model().id, rows))
            mi = []
            for bk in ids[0:min(10, len(ids))]:
                mi.append(db.get_proxy_metadata(bk))
        t = TemplateDialog(self, line_edit.text(), mi=mi, fm=db.field_metadata)
        t.setWindowTitle(_('Edit template for caption') if line_edit is self.opt_bookshelf_title_template else _('Edit template for book size'))
        if t.exec():
            line_edit.setText(t.rule[1])

    def use_pages(self):
        fm = self.gui.current_db.new_api.field_metadata
        keys = tuple(k for k in fm.all_field_keys() if fm[k].get('name'))
        names = [fm[k]['name'] for k in keys]
        try:
            idx = keys.index('#pages')
        except ValueError:
            idx = 0
        item, ok = QInputDialog.getItem(self, _('Choose a column for pages'), _(
            'Choose a column from which to get the page count for the book, such as generated by the Count Pages plugin'),
                             names, idx)
        if item and ok and item in names:
            key = keys[names.index(item)]
            template = f'''\
python:
def evaluate(book, context):
    import math
    val = book.get({key!r})
    try:
        pages = max(0, int(val))
    except Exception:
        return '0.40'
    base = 10
    return str(math.log(1+max(0, min(pages/150, base)), base+1))
'''
            self.opt_bookshelf_spine_size_template.setText(template)

    def refresh_gui(self, gui):
        gui.bookshelf_view.refresh_settings()
        gui.bookshelf_view.template_inited = False
